/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: HSpiceOut.java
 * Input/output tool: reader for HSpice output (.tr0)
 * Written by Steven M. Rubin, Sun Microsystems.
 *
 * Copyright (c) 2004 Sun Microsystems and Static Free Software
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 */
package com.sun.electric.tool.io.input;

import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.database.text.TextUtils;
import com.sun.electric.tool.simulation.Stimuli;
import com.sun.electric.tool.simulation.AnalogSignal;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Class for reading and displaying waveforms from HSpice output.
 * Thease are contained in .trX and .paX files (.pa0/.tr0, .pa1/.tr1, ...)
 */
public class HSpiceOut extends Simulate
{
	/** true if tr0 file is binary */					private boolean isTR0Binary;
	/** true if binary trX file has bytes swapped */	private boolean isTR0BinarySwapped;
	/** the "trX" file extension (could be tr0,1...) */	private String tr0Extension;
	/** the "paX" file extension (could be pa0,1...) */	private String pa0Extension;
	private int binaryTR0Size, binaryTR0Position;
	private boolean eofReached;
	private byte [] binaryTR0Buffer;

	// HSpice name associations from the .pa0 file
	private static class PA0Line
	{
		int     number;
		String  string;
	};

    private static class HSpiceStimuli extends Stimuli {
        float[][][] data;

        private void getEvent(int sweep, int index, int signalIndex, double[] result) {
            float[] d = data[sweep][index];
            result[0] = d[0];
            result[1] = result[2] = d[signalIndex + 1];
        }
    }
    
    private static class HSpiceAnalogSignal extends AnalogSignal {
        int signalIndex;

        HSpiceAnalogSignal(Stimuli sd, int signalIndex) {
            super(sd);
            this.signalIndex = signalIndex;
        } 
        
        /**
         * Method to return the value of this signal at a given event index.
         * @param sweep sweep index
         * @param index the event index (0-based).
         * @param result double array of length 3 to return (time, lowValue, highValue)
         * If this signal is not a basic signal, return 0 and print an error message.
         */
        public void getEvent(int sweep, int index, double[] result) {
            ((HSpiceStimuli)sd).getEvent(sweep, index, signalIndex, result);
        }
        
        /**
         * Method to return the number of events in one sweep of this signal.
         * This is the number of events along the horizontal axis, usually "time".
         * The method only works for sweep signals.
         * @param sweep the sweep number to query.
         * @return the number of events in this signal.
         */
        public int getNumEvents(int sweep) {
            return ((HSpiceStimuli)sd).data[sweep].length;
        }
        
        /**
         * Method to return the number of sweeps in this signal.
         * @return the number of sweeps in this signal.
         * If this signal is not a sweep signal, returns 1.
         */
        public int getNumSweeps() {
            return ((HSpiceStimuli)sd).data.length;
        }
    }
    
    HSpiceOut() {}

	/**
	 * Method to read an HSpice output file.
	 */
	protected Stimuli readSimulationOutput(URL fileURL, Cell cell)
		throws IOException
	{
		// the .paX file has name information
		List pa0List = readPA0File(fileURL);

		// show progress reading .tr0 file
		startProgressDialog("HSpice output", fileURL.getFile());

		// read the actual signal data from the .trX file
		Stimuli sd = readTR0File(fileURL, pa0List, cell);

		// stop progress dialog
		stopProgressDialog();

		// return the simulation data
		return sd;
	}

	/**
	 * Method to examine the "tr0" file pointer and read the associated "pa0" file.
	 * These files can also end in "1", "2",...
	 * As a side effect (oh no) the field variables "tr0Extension" and "pa0Extension"
	 * are set.
	 * @param fileURL the URL to the simulation output file ("pa0", "pa1", ...)
	 * @return a list of PA0Line objects that describe the name mapping file entries.
	 */
	private List readPA0File(URL fileURL)
		throws IOException
	{
		// find the associated ".pa" name file
		String tr0File = fileURL.getFile();
		tr0Extension = "";
		pa0Extension = "";
		String pa0File = tr0File + ".pa0";
		int dotPos = tr0File.lastIndexOf('.');
		if (dotPos > 0)
		{
			tr0Extension = tr0File.substring(dotPos+1);
			if (tr0Extension.length() > 2 && tr0Extension.startsWith("tr"))
			{
				pa0Extension = "pa" + tr0Extension.substring(2);
				pa0File = tr0File.substring(0, dotPos) + "." + pa0Extension;
			}
		}

		URL pa0URL = null;
		try
		{
			pa0URL = new URL(fileURL.getProtocol(), fileURL.getHost(), fileURL.getPort(), pa0File);
		} catch (java.net.MalformedURLException e)
		{
		}
		if (pa0URL == null) return null;
        if (!TextUtils.URLExists(pa0URL)) return null;
		if (openTextInput(pa0URL)) return null;

		List pa0List = new ArrayList();
		for(;;)
		{
			// get line from file
			String nextLine = lineReader.readLine();
			if (nextLine == null) break;

			// break into number and name
			String trimLine = nextLine.trim();
			int spacePos = trimLine.indexOf(' ');
			if (spacePos > 0)
			{
				// save it in a PA0Line object
				PA0Line pl = new PA0Line();
				pl.number = TextUtils.atoi(trimLine, 0, 10);
				pl.string = removeLeadingX(trimLine.substring(spacePos+1).trim());
				pa0List.add(pl);
			}
		}
		closeInput();
		return pa0List;
	}

	private Stimuli readTR0File(URL fileURL, List pa0List, Cell cell)
		throws IOException
	{
		if (openBinaryInput(fileURL)) return null;

		// get number of nodes
		int nodcnt = getHSpiceInt();

		// get number of special items
		int numnoi = getHSpiceInt();

		// get number of conditions
		int cndcnt = getHSpiceInt();

		/*
		 * Although this isn't documented anywhere, it appears that the 4th
		 * number in the file is a multiplier for the first, which allows
		 * there to be more than 10000 nodes.
		 */
		StringBuffer line = new StringBuffer();
		for(int j=0; j<4; j++) line.append((char)getByteFromFile());
		int multiplier = TextUtils.atoi(line.toString(), 0, 10);
		nodcnt += multiplier * 10000;
		int numSignals = numnoi + nodcnt - 1;

		// get version number (known to work with 9007, 9601)
		int version = getHSpiceInt();
		if (version != 9007 && version != 9601)
			System.out.println("Warning: may not be able to read HSpice files of type " + version);

		// ignore the unused/title information (4+72 characters over line break)
		for(int j=0; j<76; j++)
		{
			int k = getByteFromFile();
			if (!isTR0Binary && k == '\n') j--;
		}

		// ignore the date/time information (16 characters)
		for(int j=0; j<16; j++) getByteFromFile();

		// ignore the copywrite information (72 characters over line break)
		for(int j=0; j<72; j++)
		{
			int k = getByteFromFile();
			if (!isTR0Binary && k == '\n') j--;
		}

		// get number of sweeps
		int sweepcnt = getHSpiceInt();
		if (cndcnt == 0) sweepcnt = 0;

		// ignore the Monte Carlo information (76 characters over line break)
		for(int j=0; j<76; j++)
		{
			int k = getByteFromFile();
			if (!isTR0Binary && k == '\n') j--;
		}

		// get the type of each signal
		String [] signalNames = new String[numSignals];
		int [] signalTypes = new int[numSignals];
		for(int k=0; k<=numSignals; k++)
		{
			line = new StringBuffer();
			for(int j=0; j<8; j++)
			{
				int l = getByteFromFile();
				line.append((char)l);
				if (!isTR0Binary && l == '\n') j--;
			}
			if (k == 0) continue;
			int l = k - nodcnt;
			if (k < nodcnt) l = k + numnoi - 1;
			signalTypes[l] = TextUtils.atoi(line.toString(), 0, 10);
		}
		boolean pa0MissingWarned = false;
		for(int k=0; k<=numSignals; k++)
		{
			int j = 0;
			line = new StringBuffer();
			for(;;)
			{
				int l = getByteFromFile();
				if (l == '\n') continue;
				if (l == ' ') break;
				line.append((char)l);
				j++;
				if (version == 9007 && j >= 16) break;
			}
			int l = (j+16) / 16 * 16 - 1;
			if (version == 9007)
			{
				l = (j+15) / 16 * 16 - 1;
			}
			for(; j<l; j++)
			{
				int i = getByteFromFile();
				if (!isTR0Binary && i == '\n') { j--;   continue; }
			}
			if (k == 0) continue;

			// convert name if there is a colon in it
			int startPos = 0;
			int openPos = line.indexOf("(");
			if (openPos >= 0) startPos = openPos+1;
			for(j=startPos; j<line.length(); j++)
			{
				if (line.charAt(j) == ':') break;
				if (!TextUtils.isDigit(line.charAt(j))) break;
			}
			if (j < line.length() && line.charAt(j) == ':')
			{
				l = TextUtils.atoi(line.toString().substring(startPos), 0, 10);
				PA0Line foundPA0Line = null;
				if (pa0List == null)
				{
					if (!pa0MissingWarned)
						System.out.println("ERROR: there should be a ." + pa0Extension + " file with extra signal names");
					pa0MissingWarned = true;
				} else
				{
					for(Iterator it = pa0List.iterator(); it.hasNext(); )
					{
						PA0Line pa0Line = (PA0Line)it.next();
						if (pa0Line.number == l) { foundPA0Line = pa0Line;   break; }
					}
				}
				if (foundPA0Line != null)
				{
					StringBuffer newSB = new StringBuffer();
					newSB.append(line.substring(0, startPos));
					newSB.append(foundPA0Line.string);
					newSB.append(line.substring(j+1));
					line = newSB;
				}
			} else
			{
				if (line.indexOf(".") >= 0)
				{
					String fixedLine = removeLeadingX(line.toString());
					line = new StringBuffer();
					line.append(fixedLine);
				}
			}

			// move parenthesis from the start to the last name
			openPos = line.indexOf("(");
			if (openPos >= 0)
			{
				int lastDot = line.lastIndexOf(".");
				if (lastDot >= 0)
				{
					StringBuffer newSB = new StringBuffer();
					newSB.append(line.substring(openPos+1, lastDot+1));
					newSB.append(line.substring(0, openPos+1));
					newSB.append(line.substring(lastDot+1));
					line = newSB;
				}
			}

			if (k < nodcnt) l = k + numnoi - 1; else l = k - nodcnt;
			signalNames[l] = line.toString();
		}

		// read sweep information
		if (cndcnt != 0)
		{
			int j = 0;
			line = new StringBuffer();
			for(;;)
			{
				int l = getByteFromFile();
				if (l == '\n') continue;
				if (l == ' ') break;
				line.append((char)l);
				j++;
				if (j >= 16) break;
			}
			int l = (j+15) / 16 * 16 - 1;
			for(; j<l; j++)
			{
				int i = getByteFromFile();
				if (!isTR0Binary && i == '\n') { j--;   continue; }
			}
		}

		// read the end-of-header marker
		line = new StringBuffer();
		if (!isTR0Binary)
		{
			// finish line, ensure the end-of-header
			for(int j=0; ; j++)
			{
				int l = getByteFromFile();
				if (l == '\n') break;
				if (j < 4) line.append(l);
			}
		} else
		{
			// gather end-of-header string
			for(int j=0; j<4; j++)
				line.append((char)getByteFromFile());
		}
		if (!line.toString().equals("$&%#"))
		{
			System.out.println("HSpice header improperly terminated (got "+line.toString()+")");
			closeInput();
			return null;
		}
		resetBinaryTR0Reader();

		// setup the simulation information
		HSpiceStimuli sd = new HSpiceStimuli();
		sd.setCell(cell);
		for(int k=0; k<numSignals; k++)
		{
			AnalogSignal as = new HSpiceAnalogSignal(sd, k);
			int lastDotPos = signalNames[k].lastIndexOf('.');
			if (lastDotPos >= 0)
			{
				as.setSignalContext(signalNames[k].substring(0, lastDotPos));
				as.setSignalName(signalNames[k].substring(lastDotPos+1));
			} else
			{
				as.setSignalName(signalNames[k]);
			}
		}

		List theSweeps = new ArrayList();
		int sweepCounter = sweepcnt;
		for(;;)
		{
			// get sweep info
			if (sweepcnt > 0)
			{
				float sweepValue = getHSpiceFloat();
				if (eofReached) break;
				sd.addSweep(new Double(sweepValue));
			}

			// now read the data
			List allTheData = new ArrayList();
			for(;;)
			{
				float [] oneSetOfData = new float[numSignals+1];

				// get the first number, see if it terminates
				float time = getHSpiceFloat();
				if (eofReached) break;
				oneSetOfData[0] = time;

				// get a row of numbers
				for(int k=0; k<numSignals; k++)
				{
					float value = getHSpiceFloat();
					if (eofReached) break;
					oneSetOfData[(k+numnoi)%numSignals+1] = value;
				}
				if (eofReached) break;
				allTheData.add(oneSetOfData);
			}
			theSweeps.add(allTheData);
			sweepCounter--;
			if (sweepCounter <= 0) break;
			eofReached = false;
		}
		closeInput();
        
        // Put data to HSpiceStimuli
        sd.data = new float[theSweeps.size()][][];
        for (int i = 0; i < sd.data.length; i++) {
            ArrayList allTheData = (ArrayList)theSweeps.get(i);
            float[][] d = new float[allTheData.size()][];
            for (int j = 0; j < d.length; j++)
                d[j] = (float[])allTheData.get(j);
            sd.data[i] = d;
        }
		return sd;
	}

	/*
	 * Method to reset the binary tr0 block pointer (done between the header and
	 * the data).
	 */
	private void resetBinaryTR0Reader()
	{
		binaryTR0Size = 0;
		binaryTR0Position = 0;
	}

	/*
	 * Method to read the next block of tr0 data.  Skips the first byte if "firstbyteread"
	 * is true.  Returns true on EOF.
	 */
	private boolean readBinaryTR0Block(boolean firstbyteread)
		throws IOException
	{
		// read the first word of a binary tr0 block
		if (!firstbyteread)
		{
			if (dataInputStream.read() == -1) return true;
		}
		for(int i=0; i<3; i++)
			if (dataInputStream.read() == -1) return true;
		updateProgressDialog(4);

		// read the number of 8-byte blocks
		int blocks = 0;
		for(int i=0; i<4; i++)
		{
			int uval = dataInputStream.read();
			if (uval == -1) return true;
			if (isTR0BinarySwapped) blocks = ((blocks >> 8) & 0xFFFFFF) | ((uval&0xFF) << 24); else
				blocks = (blocks << 8) | uval;
		}
		updateProgressDialog(4);

		// skip the dummy word
		for(int i=0; i<4; i++)
			if (dataInputStream.read() == -1) return true;
		updateProgressDialog(4);

		// read the number of bytes
		int bytes = 0;
		for(int i=0; i<4; i++)
		{
			int uval = dataInputStream.read();
			if (uval == -1) return true;
			if (isTR0BinarySwapped) bytes = ((bytes >> 8) & 0xFFFFFF) | ((uval&0xFF) << 24); else
				bytes = (bytes << 8) | uval;
		}
		updateProgressDialog(4);

		// now read the data
		if (bytes > 8192)
		{
			System.out.println("ERROR: block is " + bytes + " long, but limit is 8192");
			bytes = 8192;
		}
		int amtread = dataInputStream.read(binaryTR0Buffer, 0, bytes);
		if (amtread != bytes) return true;
		updateProgressDialog(bytes);

		// read the trailer count
		int trailer = 0;
		for(int i=0; i<4; i++)
		{
			int uval = dataInputStream.read();
			if (uval == -1) return true;
			if (isTR0BinarySwapped) trailer = ((trailer >> 8) & 0xFFFFFF) | ((uval&0xFF) << 24); else
				trailer = (trailer << 8) | uval;
		}
		if (trailer != bytes) return true;
		updateProgressDialog(4);

		// set pointers for the buffer
		binaryTR0Position = 0;
		binaryTR0Size = bytes;
		return false;
	}

	/**
	 * Method to get the next character from the simulator (file or pipe).
	 * Returns EOF at end of file.
	 */
	private int getByteFromFile()
		throws IOException
	{
		if (byteCount == 0)
		{
			// start of HSpice file: see if it is binary or ascii
			int i = dataInputStream.read();
			if (i == -1) return(i);
			updateProgressDialog(1);
			if (i == 0 || i == 4)
			{
				isTR0Binary = true;
				isTR0BinarySwapped = false;
				if (i == 4) isTR0BinarySwapped = true;
				binaryTR0Buffer = new byte[8192];
				if (readBinaryTR0Block(true)) return(-1);
			} else
			{
				isTR0Binary = false;
				return(i);
			}
		}
		if (isTR0Binary)
		{
			if (binaryTR0Position >= binaryTR0Size)
			{
				if (readBinaryTR0Block(false))
					return(-1);
			}
			int val = binaryTR0Buffer[binaryTR0Position];
			binaryTR0Position++;
			return val&0xFF;
		}
		int i = dataInputStream.read();
		updateProgressDialog(1);
		return i;
	}

	private int getHSpiceInt()
		throws IOException
	{
		StringBuffer line = new StringBuffer();
		for(int j=0; j<4; j++) line.append((char)getByteFromFile());
		return TextUtils.atoi(line.toString().trim(), 0, 10);
	}

	/**
	 * Method to read the next floating point number from the HSpice file into "val".
	 * Returns positive on error, negative on EOF, zero if OK.
	 */
	private float getHSpiceFloat()
		throws IOException
	{
		if (!isTR0Binary)
		{
			StringBuffer line = new StringBuffer();
			for(int j=0; j<11; j++)
			{
				int l = getByteFromFile();
				if (l == -1)
				{
					eofReached = true;   return 0;
				}
				line.append((char)l);
				if (l == '\n') j--;
			}
			String result = line.toString();
			if (result.equals("0.10000E+31")) { eofReached = true;   return 0; }
			return (float)TextUtils.atof(result);
		}

		// binary format
		int fi0 = getByteFromFile();
		int fi1 = getByteFromFile();
		int fi2 = getByteFromFile();
		int fi3 = getByteFromFile();
		if (fi0 < 0 || fi1 < 0 || fi2 < 0 || fi3 < 0)
		{
			eofReached = true;
			return 0;
		}
		fi0 &= 0xFF;
		fi1 &= 0xFF;
		fi2 &= 0xFF;
		fi3 &= 0xFF;
		int fi = 0;
		if (isTR0BinarySwapped)
		{
			fi = (fi3 << 24) | (fi2 << 16) | (fi1 << 8) | fi0;
		} else
		{
			fi = (fi0 << 24) | (fi1 << 16) | (fi2 << 8) | fi3;
		}
		float f = Float.intBitsToFloat(fi);

		if (f > 1.00000000E30 && f < 1.00000002E30)
		{
			eofReached = true;
			return 0;
		}
		return f;
	}

}
